<?php
    
namespace App\Http\Controllers;
    
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Arr;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\Rule;
use App\Models\User;
use App\Http\Helper\Helper;
use Hash;
use DB;
use Auth;
 
class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request): View
    {
        Helper::checkPermissions('users');
        $page = 'users_';
        $page_title = __('app.users');
        $breadcrumbs = [
            __('app.user_management') => route('users.index'),
            __('app.users') => route('users.index')
        ];  
        return view('users.index',compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Retrun JSON datatable data
    */
    public function getUsers(Request $request): Void
    {
        $result = User::select('id', 'name', 'email', 'is_active', 'created_at')
          ->where('is_client', 0)
          ->where('app_id', Auth::user()->app_id)
          ->where('id', '<>', 1); // avoid to get superadmin as came with installation

        $columns = ['id', 'id', 'name', 'email', 'id', 'is_active', 'created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $users = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($users as $user) {
          $checkbox = "<input type=\"checkbox\" value=\"{$user->id}\" class=\"form-check-input\">";


          $actions = '<div class="btn-group">';
          $actions .= '<button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').'</button>';
          $actions .= '<ul class="dropdown-menu" style="">';
          $actions .= '<li><a class="dropdown-item" href="'.route('users.edit', $user->id).'"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';
          $actions .= '<li class="dropdown-item">
                <form action="'.route('user.impersonate', $user->id).'" method="POST" style="display: inline;">
                    '.csrf_field().'
                    <button type="submit" class="dropdown-item" style="background: none; border: none; padding: 0; color: inherit; cursor: pointer;">
                        <i class="bi bi-people"></i> '.__('app.impersonate').'
                    </button>
                </form>
            </li>';
          $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$user->id.'\', \''.route('users.destroy', $user->id).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
          $actions .= '</ul>';

          $status = $user->is_active 
            ? '<span class="badge bg-success-gradient">'.__('app.active').'</span>'
            : '<span class="badge bg-danger-gradient">'.__('app.in_active').'</span>';


          if(!empty($user->roles->pluck('display_name')[0])) {
            $role_id = $user->roles->pluck('id')[0];
            $user_role = "<a href='".route('roles.edit', $role_id)."' class='link-info link-offset-2 link-underline-opacity-25 link-underline-opacity-100-hover text-decoration-none'>".$user->roles->pluck('display_name')[0]."</a>";
          } else {
            $user_role = '---';
          }

          $data['data'][] = [
            "DT_RowId" => "row_{$user->id}",
            $checkbox,
            $user->id,
            $user->name,
            $user->email,
            $user_role,
            $status,
            Helper::datetimeDisplay($user->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }
    
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(): View
    {
        Helper::checkPermissions('users');
        $page = 'users_';
        $page_title = __('app.add_new_user');
        $breadcrumbs = [
            __('app.user_management') => route('users.index'),
            __('app.users') => route('users.index')
        ];
        $roles = Role::select('name','display_name')->where('type', 'user')->get();
        return view('users.create',compact('roles', 'page', 'page_title', 'breadcrumbs'));
    }
    
    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request): RedirectResponse
    {
        $this->validate($request, [
            'name' => 'required',
            'email' => [
                'required',
                'email',
                Rule::unique('users', 'email')->where('is_client', 0) // Unique only for `is_client = 0`
            ],
            'password' => 'required|same:confirm-password',
            'roles' => [
                'required',
                'not_in:'
            ],
            'is_active' => 'required|boolean',
        ]);
    
        $input = $request->all();
        $input['password'] = Hash::make($input['password']);
        $input['app_id'] = Auth::user()->app_id;
        $input['parent_id'] = Auth::user()->id;
    
        $user = User::create($input);
        $user->assignRole($request->input('roles'));

        // save log
        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.user') . " ({$request->name}) ". __('app.log_create')); // log
    
        return redirect()->route('users.index')
            ->with('success',__('app.user') .' '. __('app.log_create'));
    }
    
    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(User $user): View
    {
        Helper::checkPermissions('users');
        $page = 'users_';
        $page_title = __('app.edit_user');
        $breadcrumbs = [
            __('app.user_management') => route('users.index'),
            __('app.users') => route('users.index')
        ];

        $roles = Role::select('name','display_name')->where('type', 'user')->get();
        $userRole = $user->roles->pluck('name','name')->all();
        $profile = false;
        
        return view('users.edit',compact('user','roles','userRole', 'profile', 'page', 'page_title', 'breadcrumbs'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function profile(): View
    {
        $page = 'none'; // choose sidebar menu option
        $page_title = __('app.edit_user');
        $user = User::find(\Auth::user()->id);
        $profile = true;
        return view('users.edit',compact('user', 'profile', 'page_title'));
    }
    
    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, User $user): RedirectResponse
    {
        $this->validate($request, [
            'name' => 'required',
            'email' => [
                'required',
                'email',
                Rule::unique('users', 'email')
                    ->where('is_client', 0)
                    ->ignore($user->id) // Ignore the current user's ID
            ],
            'password' => 'nullable|same:confirm-password', // Password is optional for updates
            'roles' => [
                'required',
                'not_in:'
            ],
            'is_active' => 'required|boolean',
        ]);
    
        $input = $request->all();
        if(!empty($input['password'])){ 
            $input['password'] = Hash::make($input['password']);
        } else {
            $input = Arr::except($input,array('password'));    
        }

        $user->update($input);
        DB::table('model_has_roles')->where('model_id',$user->id)->delete();
    
        $user->assignRole($request->input('roles'));

        // save log
        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.user') . " ({$request->name}) ". __('app.log_update'));

        return redirect()->back()
            ->with('success', __('app.user') . ' ' . __('app.updated_successfully'));
    }
    
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
          $ids = array_values($request->ids);
          $names = json_encode(array_values(User::whereIn('id', $ids)->pluck('name')->toArray()));
          $destroy = User::whereIn('id', $ids)->delete();
        } else {
          $names = User::whereId($id)->value('name');
          $destroy = User::destroy($id);
        }

        activity('delete')->withProperties(['app_id' => 1])->log(__('app.user') . " ({$names}) ". __('app.log_delete'));

        $data = [
                'success' => true,
                'message' => __('app.deleted_successfully')
            ];
        return response()->json($data, 200); 
    }
}
