<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\Rule;
use App\Models\TrackingDomain;
use App\Http\Helper\Helper;
use Illuminate\View\View;
use Auth;

class TrackingDomainController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        Helper::checkPermissions('tracking_domains');
        $page = "tracking_domains";
        $page_title = __('app.tracking_domains');
        $breadcrumbs = [
            __('app.tracking_domains') => route('tracking_domains.index'),
            __('app.manage') => '#'
        ];  
        return view('tracking_domains.index',compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
       * Retrun JSON datatable data
    */
    public function getTrackingDomains(Request $request): Void
    {
    $result = TrackingDomain::select('id', 'domain', 'is_active', 'is_verified', 'created_at')
        ->app();

    $columns = ['id', 'id', 'domain', 'is_active', 'is_verified', 'created_at'];

    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $tracking_domains = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($tracking_domains as $tracking_domain) {
      $checkbox = "<input type=\"checkbox\" value=\"{$tracking_domain->id}\" class=\"form-check-input\">";

      $actions = '<div class="btn-tracking_domain">';
      $actions .= '<button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').'</button>';
      $actions .= '<ul class="dropdown-menu" style="">';
      $actions.= '<li><a class="dropdown-item" href="javascript:;" onclick="viewModal(\'modal\', \''.route('tracking_domains.edit', $tracking_domain->id).'\')"><i class="bi bi-pencil"></i> '.__('app.edit').'</a></li>';
      $actions .= '<li><a class="dropdown-item" href="javascript:;" onclick="domainVerifications(\''.$tracking_domain->id.'\', \'tracking\')"><i class="bi bi-check-lg"></i> '.__('app.verify_tracking').'</a></li>';
      $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$tracking_domain->id.'\', \''.route('tracking_domains.destroy', $tracking_domain->id).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
      $actions .= '</ul>';

      $status = $tracking_domain->is_active 
            ? "<span class='badge bg-success-gradient'>".__('app.yes')."</small>"
            : "<span class='badge bg-danger-gradient'>".__('app.no')."</small>";

      $tracking_verified = $tracking_domain->is_verified 
            ? "<span class='badge bg-success-gradient'>".__('app.verified')."</small>"
            : "<span class='badge bg-danger-gradient'>".__('app.unverified')."</small>";

      $data['data'][] = [
        "DT_RowId" => "row_{$tracking_domain->id}",
        $checkbox,
        $tracking_domain->id,
        $tracking_domain->domain,
        $status,
        "<span id='tracking-{$tracking_domain->id}'>{$tracking_verified}</span>",
        Helper::datetimeDisplay($tracking_domain->created_at),
        $actions
      ];
    }
    echo json_encode($data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        Helper::checkPermissions('tracking_domains');
        return view('tracking_domains.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'domain' => [
                'required', 
                'regex:/^([a-zA-Z0-9-]+\.)+[a-zA-Z]{2,}$/',
                Rule::unique('tracking_domains')->where(function ($query) use ($request) {
                    return $query->where('app_id', Auth::user()->app_id);
                }),
            ],
        ]);

        $data = $this->_trackingDomainData($request);
        TrackingDomain::create($data);

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.tracking_domain') . " ({$request->domain}) ". __('app.log_create'));

        return redirect()->back()
            ->with('success', $request->domain . ' ' . __('app.log_create'));
    }


    /**
     * Show the form for editing the specified resource.
     */
    public function edit(TrackingDomain $tracking_domain): View
    {
        Helper::checkPermissions('tracking_domains');
        return view('tracking_domains.edit')->with(compact('tracking_domain'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, TrackingDomain $tracking_domain): RedirectResponse
    {
        $request->validate([
          'domain' => ['required','regex:/^([a-zA-Z0-9-]+\.)+[a-zA-Z]{2,}$/']
        ]);

        $data = $this->_trackingDomainData($request);
        $tracking_domain->fill($data)->save();

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.tracking_domain') . " ({$request->domain}) ". __('app.log_update'));

        return redirect()->back()
            ->with('success', __('app.tracking_domain') . ' ' . __('app.log_update'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
          $ids = array_values($request->ids);
          $domains = json_encode(array_values(TrackingDomain::whereIn('id', $ids)->pluck('domain')->toArray()));
          $destroy = TrackingDomain::whereIn('id', $ids)->delete();
        } else {
          $domains = TrackingDomain::whereId($id)->value('domain');
          $destroy = TrackingDomain::destroy($id);
        }
        
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.tracking_domain') . " ({$domains}) ". __('app.log_delete'));

        $data = [
            'success' => true,
            'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200); 
    }

    /**
    * Retrun data for store or update
    */
    private function _trackingDomainData($request): Array
    {
        $input = $request->except('_token');
        $input['user_id'] = Auth::user()->id;
        $input['app_id'] = Auth::user()->app_id;
        return $input;
    }
}
