<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use App\Models\Package;
use App\Http\Helper\Helper;
use Auth;

class PackageController extends Controller
{
    /**
    * Retrun index view
    */
    public function index(): view
    {
        Helper::checkPermissions('clients_packages');
        $page = 'clients_packages';
        $page_title = __('app.packages');
        $breadcrumbs = [
            __('app.client_management') => '#',
            __('app.packages') => route('packages.index'),
        ]; 
        return view('packages.index')->with(compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Retrun JSON datatable data
    */
    public function getPackages(Request $request): void
    {
        $result = Package::where('app_id', Auth::user()->app_id);

        $columns = ['id', 'name', 'created_at', 'created_at', 'created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $packages = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($packages as $package) {
          $checkbox = "<input type=\"checkbox\" class=\"form-check-input\" value=\"{$package->id}\">";
          $actions = '<div class="btn-group">';
          $actions .= '<button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').'</button>';
          $actions .= '<ul class="dropdown-menu" style="">';
          $actions.= '<li><a class="dropdown-item" href="'.route('packages.edit', [$package->id]).'"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';
          $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$package->id.'\', \''.route('packages.destroy', [$package->id]).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
          $actions .= '</ul></div>';
          $attributes = json_decode($package->attributes, true);
          $data['data'][] = [
            "DT_RowId" => "row_{$package->id}",
            $checkbox,
            $package->name,
            $attributes['no_of_contacts'] == -1 ? __('app.unlimited') : $attributes['no_of_contacts'],
            $attributes['no_of_sending_servers'] == -1 ? __('app.unlimited') : $attributes['no_of_sending_servers'],
            Helper::datetimeDisplay($package->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }

    /**
    * Retrun create view
    */
    public function create(): view
    {
        Helper::checkPermissions('clients_packages'); // check user permission
        $page = 'clients_packages';
        $page_title = __('app.add_new_package');
        $breadcrumbs = [
            __('app.client_management') => '#',
            __('app.packages') => route('packages.index'),
            __('app.add_new_package') => '#',
        ];
        return view('packages.create')->with(compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Save data
    */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
          'name' => 'required|string',
        ]);
        $input = $this->_packageData($request->except('_token'));
        $input['user_id'] = Auth::user()->id;
        $input['app_id'] = Auth::user()->app_id;
        $user = Package::create($input);

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.package') . " ({$input['name']}) ". __('app.log_create'));

        return redirect()->route('packages.index')
            ->with('success', ($request->name . ' ' . __('app.created_successfully')));
    }

    /**
    * Retrun edit view
    */
    public function edit(Package $package): view
    {
        Helper::checkPermissions('clients_packages'); // check user permission
        $page = 'clients_packages';
        $page_title = __('app.edit_package');
        $breadcrumbs = [
            __('app.client_management') => '#',
            __('app.packages') => route('packages.index'),
            __('app.edit_package') => '#',
        ];
        return view('packages.edit')->with(compact('package', 'page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Update data
    */
    public function update(Request $request, $id): RedirectResponse
    {
        $request->validate([
          'name' => 'required|string',
        ]);
        $data = $this->_packageData($request->except('_token'));
        $package = Package::findOrFail($id);
        $package->fill($data)->save();

        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.package') . " ({$request->name}) ". __('app.log_update'));

        return redirect()->route('packages.edit', [$id])
            ->with('success', ($request->name . ' ' . __('app.updated_successfully')));
    }

    /**
    * Package data
    */
    private function _packageData($input)
    {
        $list_ids = !empty($input['list_ids']) ? $input['list_ids'] : null;
        $sending_server_ids = !empty($input['sending_server_ids']) ? $input['sending_server_ids'] : null;
        $input['attributes'] = json_encode([
          'list_ids' => $list_ids,
          'sending_server_ids' => $sending_server_ids,
          'no_of_contacts' => $input['no_of_contacts'],
          'no_of_sending_servers' => $input['no_of_sending_servers']
        ]);
        return $input;
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
            $ids = array_values($request->ids);
            $names = json_encode(array_values(Package::whereIn('id', $ids)->pluck('name')->toArray()));
            $destroy = Package::whereIn('id', $ids)->delete();
        } else {
            $names = Package::whereId($id)->value('name');
            $destroy = Package::destroy($id);
        }
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.package') . " ({$names}) ". __('app.log_delete'));

        $data = [
          'success' => true,
          'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200);
    }
}
