<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Models\Lists;
use App\Http\Helper\Helper;


class ListController extends Controller
{
    /**
    * Retrun a list detail
    */
    public function get(Request $request, $id): JsonResponse
    {
        $list = Lists::whereId($id)
          ->whereAppId($request->user('api')->app_id)
          ->with('group')
          ->with('customFields')
          ->first();

        if(!empty($list)) {
          if(!empty($list->customFields)) {
            $custom_fields = [];
            foreach($list->customFields as $custom_field) {
              array_push($custom_fields, $custom_field->name);
            }
          } else {
            $custom_fields = 'None';
          }
          $data = [
            'id' => $list->id,
            'name' => $list->name,
            'group_id' => $list->group->id,
            'group_name' => $list->group->name,
            'custom_fields' => $custom_fields,
            'double_optin' => $list->double_optin,
            'welcome_email' => $list->welcome_email,
            'unsub_email' => $list->unsub_email,
            'notify_email' => $list->notify_email,
            'from_name' => $list->from_name,
            'from_email' => $list->from_email,            
            'reply_email' => $list->reply_email,
            'total_contacts' => $list->total_contacts,
            'created_at' => Helper::datetimeDisplay($list->created_at)
          ];
          $status = __('app.success');
        } else {
          $status = __('app.error');
          $data = ['message' => __('app.no_record_found')];
        }

        return Helper::apiReturn($status, $data);
    }

    /**
    * Retrun lists
    */
    public function getLists(Request $request): JsonResponse
    {
        $result = Lists::whereAppId($request->user('api')->app_id)->with('group')
          ->with('customFields');

        if(!empty($request->group_id))
          $result->whereGroupId($request->group_id);

        if(!empty($request->start_from))
          $result->offset($request->start_from);

        $limit = !empty($request->limit) ? $request->limit : 100; 
        $data = $result->limit($limit)->get();

        return Helper::apiReturn(__('app.success'), $data);
    }

    /**
    * Add a list 
    */
    public function add(Request $request): JsonResponse
    { 
        $input = $this->_listData($request);

        $status = __('app.error');

        if(empty($input['name'])) {
          $data = [
            'message' => 'name ' . __('app.is_required'),
          ];
        } elseif(empty($input['group_id'])) {
          $data = [
            'message' => 'group_id ' . __('app.is_required'),
          ];
        } elseif(empty($input['from_subject'])) {
          $data = [
            'message' => 'from_subject ' . __('app.is_required'),
          ];
        } elseif(empty($input['from_name'])) {
          $data = [
            'message' => 'from_name ' . __('app.is_required'),
          ];
        } elseif(empty($input['from_email']) || !filter_var($input['from_email'], FILTER_VALIDATE_EMAIL)) {
          $data = [
            'message' => __('app.valid'). ' from_email ' . __('app.is_required'),
          ];
        } elseif(empty($input['reply_email']) || !filter_var($input['reply_email'], FILTER_VALIDATE_EMAIL)) {
          $data = [
            'message' => __('app.valid'). ' reply_email ' . __('app.is_required'),
          ];
        } else {
          if(!\App\Models\Group::whereId($input['group_id'])->whereAppId($input['app_id'])->whereTypeId(config('custom.group_lists'))->exists()) {
            $data = [
              'message' => 'group_id ' . __('app.not_exist'),
            ];
          } else {
            $list = Lists::whereAppId($input['app_id'])->whereName($input['name'])->whereGroupId($input['group_id'])->select('id', 'name')->first();
            if(!empty($list->id)) {
              $data = [
                'id' => $list->id,
                'name' => $list->name,
                'message' => __('app.list') .' '. __('app.already_exist'),
              ];
            } else {
                  try {
                    $list = Lists::create($input);
                    $input['id'] = $list->id;

                    // Attach custom fields for list
                    if(!empty($input['custom_field_ids'])) {
                      $custom_field_ids = array_values($input['custom_field_ids']);
                      $list->customFields()->attach($custom_field_ids);
                    }

                    activity('create')->withProperties(['app_id' => $input['app_id']])->log(__('app.list') . " ({$input['name']}) ". __('app.log_create'));

                    $status = __('app.success');
                    $data = [
                      'id' => $list->id,
                      'name' => $input['name']
                    ];
                  } catch (\Exception $e) {
                    $data = [
                      'message' => null,
                    ];
                  }
            }
          }
        }

        return Helper::apiReturn($status, $data);
    }

    /**
    * Update a list 
    */
    public function update(Request $request, $id): JsonResponse
    {
        $input = $this->_listData($request);

        $status = __('app.error');
        $data = [];
        
        if($request->group_id && !\App\Models\Group::whereId($input['group_id'])->whereAppId($input['app_id'])->whereTypeId(config('custom.group_lists'))->exists()) {
            $data = [
            'message' => 'group_id ' . __('app.not_exist'),
            ];
        } else {
            try {
              $list = Lists::findOrFail($id);
              $input['name'] = empty($input['name']) ? $list->name : $input['name'];
              $input['group_id'] = empty($input['group_id']) ? $list->group_id : $input['group_id'];
              $list->fill($input)->save();

              // sync custom fields
              if(!empty($input['custom_field_ids'])) {
                  $custom_field_ids = array_values($input['custom_field_ids']);
                  $list->customFields()->sync($custom_field_ids);
              }

              activity('update')->withProperties(['app_id' => $input['app_id']])->log(__('app.list') . " ({$input['name']}) ". __('app.log_update')); 

              $status = __('app.success');
              $data = [
                'id' => $list->id,
                'message' => __('app.list') .' '. __('app.log_update'),
              ];
            } catch (\Exception $e) {
              $data = [
                'message' => null,
              ];
            }
        }

        return Helper::apiReturn($status, $data);
    }

    /**
    * Return inputed data
    */
    public function _listData($request)
    {
        $input = $request->all();
        $input['name'] = !empty($input['name']) ? trim($input['name']) : null;
        $input['group_id'] = !empty($input['group_id']) ? trim($input['group_id']) : null;
        $input['user_id'] = $request->user('api')->id;
        $input['app_id'] = $request->user('api')->app_id;
        return $input;
    }

    /**
    * Delete a list 
    */
    public function destroy(Request $request, $id): JsonResponse
    {
        $name = Lists::whereAppId($request->user('api')->app_id)->whereId($id)->value('name');

        if(!empty($name)) {
          $destroy = Lists::destroy($id);
          activity('delete')->withProperties(['app_id' => $request->user('api')->app_id])->log(__('app.list') . " ({$name}) ". __('app.log_delete'));

          $status = __('app.success');
          $data = [
            'id' => $id,
            'name' => $name
          ];
        } else {
          $status = __('app.error');
          $data = ['message' => __('app.no_record_found')];
        }
        return Helper::apiReturn($status, $data);
    }
}
